<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Tables\Sys\RestTable;
use Erlage\Photogram\Constants\RequestConstants;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;

final class Request
{
    /**
     * @return Request 
     */
    final public static function instance()
    {
        static $instance = false;

        if (false === $instance)
        {
            $instance = new Request();
        }

        return $instance;
    }

    /**
     * @var array
     */
    protected $request = array();

    public function init(): self
    {
        $this -> request = array();

        /*
        |--------------------------------------------------------------------------
        | Use input stream if not-multipart form request
        |--------------------------------------------------------------------------
        */

        if ( ! System::isMultipartRequest())
        {
            $jsonContent = \file_get_contents('php://input');

            if (System::isJson($jsonContent))
            {
                $jsonParsed = \json_decode($jsonContent, true);

                foreach ($jsonParsed as $key => $content)
                {
                    $this -> request[$key] = $content;
                }
            }
            else
            {
                /*
                |--------------------------------------------------------------------------
                | fallback to check if there's something in $_POST super global
                |--------------------------------------------------------------------------
                */

                foreach ($_POST as $key => $content)
                {
                    $this -> request[$key] = System::isJson($content) ? \json_decode($content, true) : $content;
                }
            }
        }

        /*
        |--------------------------------------------------------------------------
        | Else collect payload items that're present in $_POST superglobal
        |--------------------------------------------------------------------------
        */

        else
        {
            foreach ($_POST as $key => $content)
            {
                $this -> request[$key] = System::isJson($content) ? \json_decode($content, true) : $content;
            }

            /*
            |--------------------------------------------------------------------------
            | collect attachments[files]
            |--------------------------------------------------------------------------
            */

            $this -> request[RestTable::ATTACHEMENTS] = $this -> request[RestTable::ATTACHEMENTS] ?? array();

            if (isset($_FILES[RestTable::ATTACHEMENTS]))
            {
                foreach ($_FILES[RestTable::ATTACHEMENTS]['name'] as $possibleNamespace => $possibleNamesArray)
                {
                    $attachments = array();

                    if (\is_array($possibleNamesArray))
                    {
                        foreach ($possibleNamesArray as $fieldName => $fileName)
                        {
                            $attachments[$possibleNamespace][$fieldName] = array(
                                'name'     => $_FILES[RestTable::ATTACHEMENTS]['name'][$possibleNamespace][$fieldName],
                                'type'     => $_FILES[RestTable::ATTACHEMENTS]['type'][$possibleNamespace][$fieldName],
                                'tmp_name' => $_FILES[RestTable::ATTACHEMENTS]['tmp_name'][$possibleNamespace][$fieldName],
                                'error'    => $_FILES[RestTable::ATTACHEMENTS]['error'][$possibleNamespace][$fieldName],
                                'size'     => $_FILES[RestTable::ATTACHEMENTS]['size'][$possibleNamespace][$fieldName],
                            );
                        }
                    }
                    else
                    {
                        $attachments[$possibleNamespace] = array(
                            'name'     => $_FILES[RestTable::ATTACHEMENTS]['name'][$possibleNamespace],
                            'type'     => $_FILES[RestTable::ATTACHEMENTS]['type'][$possibleNamespace],
                            'tmp_name' => $_FILES[RestTable::ATTACHEMENTS]['tmp_name'][$possibleNamespace],
                            'error'    => $_FILES[RestTable::ATTACHEMENTS]['error'][$possibleNamespace],
                            'size'     => $_FILES[RestTable::ATTACHEMENTS]['size'][$possibleNamespace],
                        );
                    }

                    $this -> request[RestTable::ATTACHEMENTS] = $attachments;
                }
            }
        }

        /*
        |--------------------------------------------------------------------------
        | init inner payload key if not collected yet
        |--------------------------------------------------------------------------
        */

        $this -> request[RequestTable::PAYLOAD] = $this -> request[RequestTable::PAYLOAD] ?? array();

        /*
        |--------------------------------------------------------------------------
        | check query parameters & add them to inner payload
        |--------------------------------------------------------------------------
        */

        foreach ($_GET as $key => $content)
        {
            if ( ! isset($this -> request[RequestTable::PAYLOAD][$key]))
            {
                $this -> request[RequestTable::PAYLOAD][$key] = $content;
            }
        }

        /*
        |--------------------------------------------------------------------------
        | collect request meta data
        |--------------------------------------------------------------------------
        */

        $this -> request[RequestTable::METHOD] = RequestConstants::METHOD_POST === $_SERVER['REQUEST_METHOD'] ? RequestConstants::METHOD_POST : RequestConstants::METHOD_GET;
        $this -> request[RequestTable::REQ_TYPE] = isset($this -> request[RequestTable::REQ_TYPE]) ? \trim((string) $this -> request[RequestTable::REQ_TYPE]) : SystemConstants::NA;
        $this -> request[RequestTable::API_VERSION] = isset($this -> request[RequestTable::API_VERSION]) ? \trim((string) $this -> request[RequestTable::API_VERSION]) : SystemConstants::NA;

        /*
        |--------------------------------------------------------------------------
        | update global state
        |--------------------------------------------------------------------------
        */

        State::setMethod($this -> request[RequestTable::METHOD]);
        State::setReqType($this -> request[RequestTable::REQ_TYPE]);
        State::setApiVersion($this -> request[RequestTable::API_VERSION]);

        /*
        |--------------------------------------------------------------------------
        | if request logger is enabled
        |--------------------------------------------------------------------------
        */

        if (MODE_LOG_REQUEST_DISPATCH)
        {
            SystemLogger::request(self::instance() -> getRequestData());
        }

        return $this;
    }

    public function getRequestData(): array
    {
        return $this -> request;
    }

    public function findKeyOffset(string $key, ...$path): string
    {
        $key = $this -> findKey($key, RequestTable::PAYLOAD, RequestTable::OFFSET, ...$path);

        if (SystemConstants::NA == $key)
        {
            return '0';
        }

        return $key;
    }

    public function findKey(string $key, ...$path): string
    {
        if (0 === \count($path))
        {
            $path = array(RequestTable::PAYLOAD);
        }

        $haystack = $this -> request;

        // traverse the path
        foreach ($path as $pathItem)
        {
            if ( ! isset($haystack[$pathItem]))
            {
                return SystemConstants::NA;
            }

            $haystack = $haystack[$pathItem];
        }

        if ( ! isset($haystack[$key]))
        {
            return SystemConstants::NA;
        }

        if (\is_array($haystack[$key]))
        {
            return \json_encode($haystack[$key]);
        }

        return \trim((string) $haystack[$key]);
    }

    public function getCollection(string $key, ...$path): array
    {
        if (0 === \count($path))
        {
            $path = array(RequestTable::PAYLOAD);
        }

        $haystack = $this -> request;

        // traverse the path
        foreach ($path as $pathItem)
        {
            if ( ! isset($haystack[$pathItem]))
            {
                return array();
            }

            $haystack = $haystack[$pathItem];
        }

        if ( ! isset($haystack[$key]))
        {
            return array();
        }

        if (\is_string($haystack[$key]))
        {
            return array($haystack[$key]);
        }

        if (\is_array($haystack[$key]))
        {
            return $haystack[$key];
        }

        return array();
    }
}
